<?php

namespace Modules\Cashback\Http\Controllers;

use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Carbon\Carbon;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Cashback\Entities\CashbackRule;
use Modules\Cashback\Entities\UserCashback;
use Modules\Cashback\Entities\UserCashbackDetail;
use Modules\Cashback\Http\Requests\CashbackRuleRequest;
use Modules\Cashback\Repositories\CashbackRuleRepository;
use Modules\CourseSetting\Entities\Category;
use Modules\CourseSetting\Entities\Course;
use Modules\Subscription\Entities\CourseSubscription;
use Yajra\DataTables\Facades\DataTables;

class CashbackController extends Controller
{
    protected $cashbackRuleRepository;

    public function __construct(CashbackRuleRepository $cashbackRuleRepository)
    {
        $this->cashbackRuleRepository = $cashbackRuleRepository;
    }


    function cashbackPageCommon()
    {
        $data = [];
        $query = Course::query();
        $query->where('status', 1);
        if (isModuleActive('UpcomingCourse')) {
            $query->where('is_upcoming_course', 0);
        }
        $all_courses = $query->select('id', 'title', 'type')->get();

        $data['courses'] = $all_courses->where('type', 1);
        $data['quizzes'] = $all_courses->where('type', 2);
        $data['live_classes'] = $all_courses->where('type', 3);
        $data['categories'] = Category::where('status', 1)->select('id', 'name')->get();
        $data['instructors'] = User::where('status', 1)->whereIn('role_id', [1, 2])->select('id', 'name')->get();
        $data['students'] = User::where('status', 1)->where('role_id', 3)->select('id', 'name')->get();
        $data['subscription_plans'] = [];
        $data['upcoming_courses'] = [];
        if (isModuleActive('Subscription')) {
            $data['subscription_plans'] = CourseSubscription::where('status', 1)->select('id', 'title')->get();
        }
        if (isModuleActive('UpcomingCourse')) {
            $data['upcoming_courses'] = $all_courses->where('type', 1)->where('is_upcoming_course', 1);
        }
        return $data;
    }

    public function index()
    {
        $data = [];
        $data['table_columns'] = $this->cashbackRuleRepository->tableColumns();
        $data['rules'] = $this->cashbackRuleRepository->all(['*'], ['rule_specific_item', 'userCashbackDetails']);
        $data['title'] = _trans('cashback.Cashback Rules List');
        return view('cashback::rule.index', compact('data'));
    }

    public function userCashbackStatus($user_id)
    {
        try {
            $user = User::find($user_id);
            if ($user->is_cashback_enabled == 1) {
                $user->is_cashback_enabled = 0;
                $type = _trans('cashback.Disabled');
            } else {
                $user->is_cashback_enabled = 1;
                $type = _trans('cashback.Enabled');
            }
            $user->save();
            Toastr::success(_trans('cashback.User Cashback Status ' . $type, _trans('common.Success')));
            return redirect()->back();
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }


    public function create()
    {
        $data = [];
        $data = $this->cashbackPageCommon();
        $data['title'] = _trans('cashback.Create Cashback Rule');
        return view('cashback::rule.create', compact('data'));
    }

    public function edit($id)
    {
        $data = [];
        $data = $this->cashbackPageCommon();
        $data['title'] = _trans('cashback.Edit Cashback Rule');
        $data['rule'] = $this->cashbackRuleRepository->findById($id, ['*'], ['rule_specific_item', 'users']);
        return view('cashback::rule.create', compact('data'));
    }


    public function store(CashbackRuleRequest $request)
    {
        try {
            $result = $this->cashbackRuleRepository->store($request);
            if ($result) {
                Toastr::success(_trans('cashback.Rule Saved successful'), trans('common.Success'));
                return redirect()->route('cashback.rules');
            } else {
                Toastr::error(_trans('cashback.Rule Not Saved'), trans('common.Error'));
                return redirect()->back();
            }
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function update(CashbackRuleRequest $request)
    {
        try {
            $result = $this->cashbackRuleRepository->updateRule($request);
            if ($result) {
                Toastr::success(_trans('cashback.Rule Saved successful'), trans('common.Success'));
                return redirect()->route('cashback.rules');
            } else {
                Toastr::error(_trans('cashback.Rule Not Saved'), trans('common.Error'));
                return redirect()->back();
            }
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }


    public function transaction()
    {
        $data = [];
        $data['title'] = _trans('cashback.Cashback Transaction List');

        return view('cashback::transaction.index', compact('data'));
    }


    public function transactionLoad(Request $request)
    {
        try {
            $query = UserCashbackDetail::query();
            $query->with(['user_cashback', 'course_info', 'subscription_info']);
            if ($request->name) {
                $query->whereHas('user_cashback.user', function ($q) use ($request) {
                    $q->where('name', 'like', '%' . $request->name . '%');
                });
            }
            if ($request->target_type != '') {
                if ($request->target_type == 'gift') {
                    $query->where('is_gift', 1);
                }
                if (!in_array($request->target_type, ['gift', 'all'])) {
                    $query->where('product_type', $request->target_type);
                }
            }
            if ($request->status != '') {
                $query->where('status', $request->status);
            }
            if ($request->startDate != '' && $request->endDate != '') {
                $startDate = Carbon::parse($request->startDate)->startOfDay();
                $endDate = Carbon::parse($request->endDate)->endOfDay();
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }

            $query->latest();


            return Datatables::of($query)
                ->addIndexColumn()
                ->addColumn('name', function ($query) {
                    return @$query->user_cashback->user->name;
                })
                ->editColumn('image', function ($query) {
                    $query = @$query->user_cashback->user;
                    return view('backend.partials._td_image', compact('query'));
                })
                ->editColumn('product', function ($query) {
                    $product_name = '';
                    if (in_array($query->product_type, ['course', 'quiz', 'live_class'])) {
                        $product_name = @$query->course_info->getTranslation('title', auth()->user()->language_code ?? 'en');
                    } elseif ($query->product_type == 'subscription') {
                        $product_name = isset($query->subscription_info) ? $query->subscription_info->course->getTranslation('title', auth()->user()->language_code ?? 'en') : _trans('cashback.Not found');
                    } elseif ($query->product_type == 'recharge') {
                        $product_name = _trans('cashback.Recharge');
                    } else {
                        $product_name = _trans('cashback.Not found');
                    }
                    return $product_name;
                })
                ->editColumn('description', function ($query) {
                    $description = '';
                    if (in_array($query->product_type, ['course', 'quiz', 'live_class'])) {
                        $description = _trans('cashback.Cashback on course purchase');
                    } elseif ($query->product_type == 'subscription') {
                        $description = _trans('cashback.Cashback on subscription purchase');
                    } elseif ($query->product_type == 'recharge') {
                        $description = _trans('cashback.Cashback on recharge amount');
                    } else {
                        $description = _trans('cashback.Cashback');
                    }
                    return $description;
                })
                ->editColumn('amount', function ($query) {
                    return getPriceFormat(@$query->purchase_amount);
                })
                ->editColumn('cashback_amount', function ($query) {
                    return getPriceFormat(@$query->cashback_amount);
                })
                ->editColumn('date', function ($query) {
                    return showDate(@$query->created_at);
                })
                ->editColumn('status', function ($query) {

                    return ucfirst(@$query->status);
                })
                ->editColumn('action', function ($query) {
                    if ($query->status == 'paid') {
                        return view('cashback::transaction.action_menu', compact('query'));
                    }
                })->rawColumns(['name'])
                ->make(true);
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function history()
    {
        $data = [];
        $data['title'] = _trans('cashback.Cashback Transaction History');

        return view('cashback::transaction.history', compact('data'));
    }


    public function historyLoad(Request $request)
    {
        try {
            $query = UserCashback::query();
            $query->with(['user']);
            if ($request->name) {
                $query->whereHas('user', function ($q) use ($request) {
                    $q->where('name', 'like', '%' . $request->name . '%');
                });
            }
            if ($request->startDate != '' && $request->endDate != '') {
                $startDate = Carbon::parse($request->startDate)->startOfDay();
                $endDate = Carbon::parse($request->endDate)->endOfDay();
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }

            $query->latest();


            return Datatables::of($query)
                ->addIndexColumn()
                ->addColumn('name', function ($query) {
                    return @$query->user->name;
                })
                ->editColumn('image', function ($query) {
                    $query = @$query->user;
                    return view('backend.partials._td_image', compact('query'));
                })
                ->editColumn('total_purchase', function ($query) {
                    return getPriceFormat(@$query->purchase_amount);
                })
                ->editColumn('total_cashback', function ($query) {
                    return getPriceFormat(@$query->cashback_amount);
                })
                ->editColumn('last_cashback', function ($query) {
                    return showDate(@$query->last_cashback_datetime);
                })
                ->editColumn('action', function ($query) {
                    return view('cashback::transaction.history_menu', compact('query'));
                })->rawColumns(['name'])
                ->make(true);
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function transactionRefund($trans_id)
    {
        try {
            $transaction = UserCashbackDetail::with('user_cashback')->find($trans_id);

            $user = User::find($transaction->user_cashback->user_id);
            if ($user->balance >= $transaction->cashback_amount) {
                $user->balance -= $transaction->cashback_amount;
                $user->save();

                $transaction->status = 'refunded';
                $transaction->save();
                Toastr::success(_trans('cashback.Cashback amount refunded', _trans('common.Success')));
                return redirect()->back();
            }
            Toastr::error(_trans('cashback.User have no Sufficient balance'), _trans('common.Error'));
            return redirect()->back();
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function destroy(Request $request)
    {
        try {
            $cashback_rule = CashbackRule::find($request->id);
            $cashback_rule->delete();
            Toastr::success(_trans('cashback.Cashback Rule Deleted', _trans('common.Success')));
            return redirect()->back();
        } catch (\Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }
}
