<?php

namespace Modules\Installment\Repositories;

use App\Repositories\Eloquents\BaseRepository;
use App\Traits\UploadMedia;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Modules\Installment\Entities\InstallmentPlan;
use Modules\Installment\Entities\InstallmentPlanPaymentStep;
use Modules\Installment\Entities\PlanSpecificItem;

class InstallmentPlanRepository extends BaseRepository
{
    use UploadMedia;

    protected $model;

    public function __construct(InstallmentPlan $model)
    {
        $this->model = $model;
    }

    function tableColumns()
    {
        return [
            _trans('common.SL'),
            _trans('common.Title'),
            _trans('installment.Sales'),
            _trans('installment.Upfront'),
            _trans('installment.Total Installment'),
            _trans('installment.Installment Amount'),
            _trans('installment.Capacity'),
            _trans('installment.Start Date'),
            _trans('installment.End date'),
            _trans('common.Status'),
            _trans('common.Action'),
        ];
    }

    function planFeature($request, $plan_id)
    {
        try {
            $plan = InstallmentPlan::find($plan_id);

            $plan->features()->delete();
            if ($request->features) {
                foreach ($request->features as $feature) {
                    $plan->features()->create([
                        'title' => $feature,
                        'installment_plan_id' => $plan_id,
                    ]);
                }
            }
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

    function planTargetDetails($request, $plan_id, $array_name, $target_type, $plan_specific_item)
    {
        try {
            $delete_previous = $plan_specific_item->plan_specific_item_details_relation()->delete();
            if (!$request->$array_name) return true;
            foreach ($request->$array_name as $target_id) {
                $plan_specific_item->plan_specific_item_details_relation()->create([
                    'item_id' => $target_id,
                    'type' => $target_type,
                ]);
            }
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

    function planTarget($request, $plan_id)
    {
        try {
            $plan = InstallmentPlan::find($plan_id);

            $plan->plan_specific_item()->delete();

            $plan_specific_item = new PlanSpecificItem();
            $plan_specific_item->installment_plan_id = $plan_id;
            $plan_specific_item->target_type = $request->target_type;
            $plan_specific_item->target = $request->target;
            $plan_specific_item->save();

            $this->planTargetDetails($request, $plan_id, $plan_specific_item->target, $plan_specific_item->target, $plan_specific_item);

            return true;
        } catch (\Throwable $th) {
            return false;
        }

    }

    function planUpfrontSetup($request, $plan_id)
    {
        try {
            $plan = InstallmentPlan::find($plan_id);
            $plan->paymentSteps()->delete();
            $payment_step = new InstallmentPlanPaymentStep();
            $payment_step->installment_plan_id = $plan_id;
            $payment_step->payment_title = 'Upfront';
            $payment_step->payment_deadline = null;
            $payment_step->payment_amount = $request->upfront;
            $payment_step->amount_type = $request->upfront_type;
            $payment_step->save();
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

    function planPaymentStep($request, $plan_id)
    {
        try {
            $plan = $this->model->find($plan_id);
            if ($request->payment_steps) {
                foreach ($request->payment_steps as $step) {
                    $payment_step = new InstallmentPlanPaymentStep();
                    $payment_step->installment_plan_id = $plan_id;
                    $payment_step->payment_title = $step['payment_title'];
                    $payment_step->payment_deadline = $step['payment_deadline'];
                    $payment_step->payment_amount = $step['payment_amount'];
                    $payment_step->amount_type = $step['amount_type'];
                    $payment_step->save();
                }
            }
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

    function store($request)
    {

        try {
            DB::beginTransaction();

            $plan = new $this->model;
            foreach ($request->title as $key => $name) {
                $plan->setTranslation('title', $key, $name);
            }

            foreach ($request->description as $key => $description) {
                $plan->setTranslation('description', $key, $description);
            }
            $plan->capacity = $request->capacity;


            $start_time = Carbon::createFromFormat('h:i A', $request->start_time)->format('H:i');
            $plan->start_datetime = Carbon::createFromFormat('d-m-Y H:i', Carbon::parse($request->start_date)->format('d-m-Y') . " " . $start_time);

            if ($request->end_date) {
                $end_time = Carbon::createFromFormat('h:i A', $request->end_time)->format('H:i');
                $plan->end_datetime = Carbon::createFromFormat('d-m-Y H:i', Carbon::parse($request->end_date)->format('d-m-Y') . " " . $end_time);
            } else {
                $plan->end_datetime = null;
            }


            $plan->verification_required = $request->verification_required;
            $plan->verification_description = $request->verification_description;

            $plan->request_uploads = $request->request_uploads;
            $plan->bypass_verification_for_verified_users = $request->bypass_verification_for_verified_users;

            $plan->status = $request->status ?? 1;
            $plan->save();

            if ($request->banner) {
                $plan->banner = $this->generateLink($request->banner, $plan->id, get_class($plan), 'banner');
            }

            if ($request->verification_banner) {
                $plan->verification_banner = $this->generateLink($request->verification_banner, $plan->id, get_class($plan), 'verification_banner');
            }

            if ($request->verification_video) {
                $plan->verification_video = $this->generateLink($request->verification_video, $plan->id, get_class($plan), 'verification_video');
            }
            $plan->save();

            $this->planTarget($request, $plan->id);
            $this->planFeature($request, $plan->id);
            $this->planUpfrontSetup($request, $plan->id);
            $this->planPaymentStep($request, $plan->id);

            DB::commit();
            return $plan;
        } catch (\Throwable $th) {
            DB::rollBack();
            return false;
        }


    }

    function updatePlan($request)
    {

        try {
            DB::beginTransaction();

            $plan = InstallmentPlan::find($request->id);
            foreach ($request->title as $key => $name) {
                $plan->setTranslation('title', $key, $name);
            }

            foreach ($request->description as $key => $description) {
                $plan->setTranslation('description', $key, $description);
            }

            $plan->capacity = $request->capacity;

            $upload_path = 'public/uploads/installment';

            $start_time = Carbon::createFromFormat('h:i A', $request->start_time)->format('H:i');
            $plan->start_datetime = Carbon::createFromFormat('d-m-Y H:i', Carbon::parse($request->start_date)->format('d-m-Y') . " " . $start_time);

            if ($request->end_date) {
                $end_time = Carbon::createFromFormat('h:i A', $request->end_time)->format('H:i');
                $plan->end_datetime = Carbon::createFromFormat('d-m-Y H:i', Carbon::parse($request->end_date)->format('d-m-Y') . " " . $end_time);
            } else {
                $plan->end_datetime = null;
            }

            $plan->verification_required = $request->verification_required;
            $plan->verification_description = $request->verification_description;

            $plan->request_uploads = $request->request_uploads;
            $plan->bypass_verification_for_verified_users = $request->bypass_verification_for_verified_users;

            $plan->status = (int)$request->status;

            $plan->banner = null;
            $plan->verification_banner = null;
            $plan->verification_video = null;

            $plan->save();

            $this->removeLink($plan->id, get_class($plan));

            if ($request->banner) {
                $plan->banner = $this->generateLink($request->banner, $plan->id, get_class($plan), 'banner');
            }

            if ($request->verification_banner) {
                $plan->verification_banner = $this->generateLink($request->verification_banner, $plan->id, get_class($plan), 'verification_banner');
            }

            if ($request->verification_video) {
                $plan->verification_video = $this->generateLink($request->verification_video, $plan->id, get_class($plan), 'verification_video');
            }
            $plan->save();

            $this->planTarget($request, $plan->id);
            $this->planFeature($request, $plan->id);
            $this->planUpfrontSetup($request, $plan->id);
            $this->planPaymentStep($request, $plan->id);

            DB::commit();
            return true;
        } catch (\Throwable $th) {
            DB::rollBack();
            return false;
        }


    }

    function deletePlan($request)
    {
        try {
            $plan = InstallmentPlan::find($request->id);
            $plan->delete();
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

}
