<?php

namespace Modules\Appointment\Repositories\Eloquents;

use App\Repositories\Eloquents\BaseRepository;
use App\Traits\SendNotification;
use App\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Modules\Appointment\Entities\Booking;
use Modules\Appointment\Entities\BookTrailLesson;
use Modules\Appointment\Entities\Schedule;
use Modules\Appointment\Repositories\Interfaces\BookingRepositoryInterface;
use Modules\Appointment\Repositories\Interfaces\InstructorRepositoryInterface;
use Modules\Appointment\Repositories\Interfaces\TimeSlotRepositoryInterface;

class BookingRepository extends BaseRepository implements BookingRepositoryInterface
{
    use SendNotification;

    protected $user;
    protected $schedule;
    protected $timeSlotRepository;
    protected $bookTrailLessonModel;
    protected $instructorRepository;

    public function __construct(
        Booking                       $model,
        User                          $user,
        Schedule                      $schedule,
        BookTrailLesson               $bookTrailLessonModel,
        InstructorRepositoryInterface $instructorRepository,
        TimeSlotRepositoryInterface   $timeSlotRepository
    )
    {
        parent::__construct($model);
        $this->user = $user;
        $this->schedule = $schedule;
        $this->timeSlotRepository = $timeSlotRepository;
        $this->bookTrailLessonModel = $bookTrailLessonModel;
        $this->instructorRepository = $instructorRepository;
        $this->timeSlotRepository = $timeSlotRepository;
    }

    public function index(array $payload): array
    {
        if (!$payload) {
            $payload['instructor'] = '';
            $payload['time_slot'] = '';
            $payload['start_date'] = '';
            $payload['end_date'] = '';
            $payload['status'] = 0;
        }
        $data['booking_list'] = $this->filterBookedUser($payload);
        $data['currentStudent'] = $this->model->groupBy([
            'user_id',
            'instructor_id',
            'schedule_id'
        ])->where('status', 1)->count();
        $data['payload'] = $payload;
        $data += $this->timeSlotRepository->index();
        return $data;
    }

    public function bookedUser(array $payload): array
    {
        $date = gv($payload, 'date');
        $time_slot = gv($payload, 'time_slot');
        $instructor_id = gv($payload, 'instructor_id');
        $data = [];
        if ($date) {
            $data['bookedUsers'] = $this->model->with('schedule')
                ->whereHas('schedule', function ($query) use ($date) {
                    $query->where('schedule_date', $date);
                })->when($time_slot, function ($query) use ($time_slot) {
                    $query->whereHas('schedule', function ($query) use ($time_slot) {
                        $query->where('slot_id', $time_slot);
                    });
                })->when($instructor_id, function ($query) use ($instructor_id) {
                    $query->where('instructor_id', $instructor_id);
                })->where('status', 1)->get();
            $payload['instructor'] = $instructor_id;
            $payload['start_date'] = '';
            $payload['end_date'] = '';
        } elseif ($payload) {
            $data['bookedUsers'] = $this->filterBookedUser($payload);
        }
        $data['payload'] = $payload;
        $data += $this->timeSlotRepository->index();
        return $data;
    }

    private function filterBookedUser(array $payload)
    {
        $start_date = $payload['start_date'] ? date('Y-m-d', strtotime($payload['start_date'])) : '';
        $end_date = $payload['end_date'] ? date('Y-m-d', strtotime($payload['end_date'])) : '';
        $today = date('Y-m-d');
        $scheduleIds = $this->schedule->when(gv($payload, 'instructor'), function ($query) use ($payload) {
            $query->where('user_id', gv($payload, 'instructor'));
        }, function ($query) {
            $query->when(auth()->user()->role_id != 1, function ($subQuery) {
                $subQuery->where('user_id', auth()->user()->id);
            });
        })->when($start_date && !$end_date, function ($query) use ($start_date) {
            $query->where('schedule_date', $start_date);
        })->when($start_date && $end_date, function ($query) use ($start_date, $end_date) {
            $query->whereBetween('schedule_date', [$start_date, $end_date]);
        })->when($payload['time_slot'], function ($query) use ($payload) {
            $query->where('slot_id', $payload['time_slot']);
        })
            ->when(!$start_date || !$end_date, function ($query) {
            $query->where('schedule_date', '>=', date('Y-m-d'));
        })
            ->pluck('id')->toArray();
        return $bookedUsers = $this->model
            ->whereIn('schedule_id', $scheduleIds)
            ->get();
    }

    public function shareLink(array $payload)
    {
        if (gv($payload, 'type') == 'all') {
            $this->model->where('schedule_id', gv($payload, 'schedule_id'))->update([
                'type' => gv($payload, 'type'),
                'share_link' => gv($payload, 'share_link'),
                'note' => gv($payload, 'note')
            ]);
            $booking = $this->model->where('schedule_id', gv($payload, 'schedule_id'));
            $firstBooked = $booking->first();

            if (gv($payload, 'type') == 'single') {
                $firstBooked = $booking->where('id', gv($payload, 'booking_id'))
                    ->where('user_id', gv($payload, 'student_id'))->first();
            }
            $bookings = $booking->get();
            $start_time = date('h:i A', strtotime($firstBooked->schedule->slotInfo->start_time));
            $end_time = date('h:i A', strtotime($firstBooked->schedule->slotInfo->end_time));
            $timeSlot = $firstBooked->schedule->schedule_date . '(' . $start_time . '-' . $end_time . ')';
            $shortCodes = [
                'date' => $firstBooked->schedule->schedule_date,
                'link' => gv($payload, 'share_link'),
                'timeSlot' => $timeSlot,
                'note' => gv($payload, 'note')
            ];
            foreach ($bookings as $booked) {
                $shortCodes['student_name'] = $booked->userInfo->name;
                $this->shareLinkNotificationEmail($booked, $shortCodes);
            }

        }
        if (gv($payload, 'type') == 'single') {
            $this->model->where('id', gv($payload, 'booking_id'))
                ->where('user_id', gv($payload, 'student_id'))->update([
                    'type' => gv($payload, 'type'),
                    'share_link' => gv($payload, 'share_link'),
                ]);
            $booking = $this->model->where('id', gv($payload, 'booking_id'));
            $firstBooked = $booking->first();
            $shortCodes['student_name'] = $firstBooked->userInfo->name;
            $this->shareLinkNotificationEmail($firstBooked, $shortCodes);
        }
    }

    private function shareLinkNotificationEmail($booked, $shortCodes)
    {
        $this->sendNotification('Share_Link', $booked->userInfo, $shortCodes);
        $this->sendNotification('Book_Trail_Lesson', $booked->userInfo, $shortCodes);
    }

    public function bookTrailLesson($slug)
    {
        if (auth()->check()) {
            $data = $this->instructorRepository->instructorBySlug($slug);
            $instructor_id = $data['instructor']['id'];
            $bookedTrailLesson = $this->bookTrailLessonModel
                ->where('instructor_id', $instructor_id)
                ->where('student_id', auth()->user()->id)
                ->where('status', 1)
                ->first();
            if (!$bookedTrailLesson) {
                $bookedTrailLesson = $this->bookTrailLessonModel;
            }
            $bookedTrailLesson->instructor_id = $instructor_id;
            $bookedTrailLesson->student_id = auth()->user()->id;
            $bookedTrailLesson->save();
            $admin = $this->user->where('role_id', 1)->first();
            $shortCodes = [
                'student' => auth()->user()->name,
                'instructor' => $data['instructor']['name'],
            ];
            if (!$bookedTrailLesson) {

                $this->sendNotification('Book_Trail_Lesson_Admin', $admin, $shortCodes);
                $this->sendNotification('Book_Trail_Lesson_Instructor', $data['instructor'], [
                    'student' => auth()->user()->name,
                ]);

                $this->sendNotification('Book_Trail_Lesson', auth()->user(), [
                    'instructor' => $data['instructor']['name']
                ]);
            }

        } else {
            session(['redirectTo' => route('book-trail-lesson', [$slug])]);
            return \redirect(route('login'));
        }
    }

    public function trailLesson(array $payload): array
    {
        if (!$payload) {
            $payload['instructor'] = '';
            $payload['time_slot'] = '';
            $payload['start_date'] = '';
            $payload['end_date'] = '';
            $payload['status'] = 1;
        }

        if ($payload) {
            $start_date = $payload['start_date'] ? date('Y-m-d', strtotime($payload['start_date'])) : '';
            $end_date = $payload['end_date'] ? date('Y-m-d', strtotime($payload['end_date'])) : '';
            $data['bookTrailLessons'] = $this->bookTrailLessonModel
                ->when(gv($payload ?? [], 'instructor'), function ($query) use ($payload) {
                    $query->where('instructor_id', gv($payload, 'instructor'));
                }, function ($query) {
                    $query->when(auth()->user()->role_id != 1, function ($subQuery) {
                        $subQuery->where('instructor_id', auth()->id());
                    });
                })
                ->when($start_date && !$end_date, function ($query) use ($start_date) {
                    $query->whereDate('created_at', $start_date);
                })
                ->when($start_date && $end_date, function ($query) use ($start_date, $end_date) {

                    $start_date = Carbon::parse($start_date)->startOfDay()->toDateTimeString();
                    $end_date =  Carbon::parse($end_date)->endOfDay()->toDateTimeString();
                    $query->whereBetween('created_at', [$start_date, $end_date]);
                })
                ->when(gv($payload ?? [], 'time_slot'), function ($query) use ($payload) {
                    $query->whereHas('schedule', function ($subQuery) use ($payload) {
                        $subQuery->where('slot_id', gv($payload, 'time_slot'));
                    });
                })
                ->where('status', gv($payload, 'status', '0'))
                ->latest()
                ->get();

            $data['payload'] = $payload;
        }
        $data += $this->timeSlotRepository->index();
        return $data;
    }

    public function trailLessonDelete($id)
    {
        $bookTrailLessonModel = $this->bookTrailLessonModel->where('id', $id)->first();
        $shortCodes = [
            'time' => \Illuminate\Support\Carbon::now()->format('d-M-Y ,s:i A'),
            'instructor' => $bookTrailLessonModel->instructorInfo->name,
        ];

        $this->sendNotification('Book_Trail_Lesson_Rejected', $bookTrailLessonModel->userInfo, $shortCodes);

        $bookTrailLessonModel->delete();
    }

    public function create(array $payload): ?Model
    {
        return null;
    }

    public function approve($approve_id)
    {
        $detail = $this->findById($approve_id);
        $detail->update(['status' => 1]);
        $start_time = date('h:i A', strtotime($detail->schedule->slotInfo->start_time));
        $end_time = date('h:i A', strtotime($detail->schedule->slotInfo->end_time));
        $timeSlot = $detail->schedule->schedule_date . '(' . $start_time . '-' . $end_time . ')';

        $this->sendNotification('Appointment_Approve', $detail->userInfo, [
            'time' => \Illuminate\Support\Carbon::now()->format('d-M-Y ,s:i A'),
            'timeSlot' => $timeSlot,
            'student_name' => $detail->userInfo->name,
        ]);
    }

    public function reject($reject_id)
    {
        $detail = $this->findById($reject_id);
        $start_time = date('h:i A', strtotime($detail->schedule->slotInfo->start_time));
        $end_time = date('h:i A', strtotime($detail->schedule->slotInfo->end_time));
        $timeSlot = $detail->schedule->schedule_date . '(' . $start_time . '-' . $end_time . ')';

        $this->sendNotification('Appointment_Reject', $detail->userInfo, [
            'time' => \Illuminate\Support\Carbon::now()->format('d-M-Y ,s:i A'),
            'timeSlot' => $timeSlot,
            'student_name' => $detail->userInfo->name,
        ]);
        $balance = $detail->userInfo->balance + $detail->purchase_price;
        User::where('id', $detail->user_id)->update([
            'balance' => $balance,
        ]);
        $detail->delete();
    }

    public function store(array $payload)
    {

    }

    public function editValue(array $payload): array
    {
        $data = [];
        return $data;
    }

    public function update(int $modelId, array $payload): bool
    {
        return true;
    }

    public function getListByUser($user_id)
    {
        return $this->bookTrailLessonModel->where('student_id', $user_id)->where('status', 1)->get();
    }
}
