<?php

namespace Modules\Installment\Http\Controllers;

use App\Jobs\SendNotification;
use COM;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use App\Jobs\SendGeneralEmail;
use Illuminate\Routing\Controller;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Support\Facades\Auth;
use Throwable;
use Yajra\DataTables\Facades\DataTables;
use Modules\CourseSetting\Entities\Course;
use Illuminate\Contracts\Support\Renderable;
use Modules\CourseSetting\Entities\Category;
use Modules\Installment\Entities\InstallmentPlan;
use Modules\Installment\Http\Requests\PlanRequest;
use Modules\Installment\Entities\InstallmentPurchasePayment;
use Modules\Installment\Entities\InstallmentPurchaseRequest;
use Modules\Installment\Repositories\InstallmentPlanRepository;

class InstallmentController extends Controller
{
    protected $installmentPlanRepository;

    public function __construct(InstallmentPlanRepository $installmentPlanRepository)
    {
        $this->installmentPlanRepository = $installmentPlanRepository;
    }

    public function index()
    {
        $data = [];
        $data['table_columns'] = $this->installmentPlanRepository->tableColumns();
        $data['plans'] = $this->installmentPlanRepository->all(['*'], ['plan_specific_item', 'features', 'paymentSteps']);
        $data['title'] = _trans('installment.Installment Plan List');
        return view('installment::plan.index', compact('data'));
    }

    public function create()
    {
        $data = [];
        $data = $this->planPageCommon();
        $data['title'] = _trans('installment.Add New Plan');
        return view('installment::plan.create', compact('data'));
    }

    function planPageCommon()
    {
        $data = [];
        $all_courses = Course::where('status', 1)->select('id', 'title', 'type')->get();

        $data['courses'] = $all_courses->where('type', 1);
        $data['quizzes'] = $all_courses->where('type', 2);
        $data['live_classes'] = $all_courses->where('type', 3);
        $data['categories'] = Category::where('status', 1)->select('id', 'name')->get();
        $data['instructors'] = User::where('status', 1)->where('role_id', 2)->select('id', 'name')->get();
        return $data;
    }

    public function store(PlanRequest $request)
    {

        try {
            $result = $this->installmentPlanRepository->store($request);
            if ($result) {
                Toastr::success(_trans('installment.Plan Saved successful'), trans('common.Success'));
                return redirect()->route('installment.plan.list');
            } else {
                Toastr::error(_trans('installment.Plan Not Saved'), trans('common.Error'));
                return redirect()->back();
            }
        } catch (Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }


    public function show($id)
    {
        return view('installment::show');
    }


    public function edit($id)
    {
        $data = [];
        $data = $this->planPageCommon();
        $data['title'] = _trans('installment.Edit Plan');
        $data['plan'] = $this->installmentPlanRepository->findById($id, ['*'], ['plan_specific_item', 'features', 'paymentSteps']);
        return view('installment::plan.create', compact('data'));
    }


    public function update(PlanRequest $request)
    {
        try {
            $result = $this->installmentPlanRepository->updatePlan($request);
            if ($result) {
                Toastr::success(_trans('installment.Plan Updated successful'), trans('common.Success'));
                return redirect()->route('installment.plan.list');
            } else {
                Toastr::error(_trans('installment.Plan Not Updated'), trans('common.Error'));
                return redirect()->back();
            }
        } catch (Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }


    public function destroy(Request $request)
    {
        try {
            $result = $this->installmentPlanRepository->deletePlan($request);
            if ($result) {
                Toastr::success(_trans('installment.Plan Deleted successful'), trans('common.Success'));
                return redirect()->route('installment.plan.list');
            } else {
                Toastr::error(_trans('installment.Plan Not Deleted'), trans('common.Error'));
                return redirect()->back();
            }
        } catch (Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function verifiedUsers()
    {
        try {
            $data = [];
            $data['title'] = _trans('installment.Verified Users');
            return view('installment::verification.users', compact('data'));

        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function userInstallments($user_id)
    {
        try {
            $data = [];
            $data['title'] = _trans('installment.User Installments');

            $data['purchase_requests'] = InstallmentPurchaseRequest::where('payment_status', '!=', 'incomplete')
                ->where('user_id', $user_id)
                ->with('installmentPayments', 'plan', 'topic')->get();
            $data['table_columns'] = $this->usersTableColumns();
            return view('installment::verification.user_installments', compact('data'));

        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    function usersTableColumns()
    {
        return [
            _trans('installment.SL'),
            _trans('installment.Plan'),
            _trans('installment.Topic'),
            _trans('installment.Purchase Date'),
            _trans('installment.Total Amount'),
            _trans('installment.Upfront'),
            _trans('installment.Installment'),
            _trans('installment.Overdue'),
            _trans('installment.Overdue Amount'),
            _trans('installment.Remaining Days'),
            _trans('installment.Status'),
            _trans('installment.Action'),
        ];
    }

    public function overDue()
    {
        try {
            $data = [];
            $data['title'] = _trans('installment.Overdue Installment');
            return view('installment::overdue.list', compact('data'));
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function overDueLoad(Request $request)
    {

        $query = InstallmentPurchasePayment::query();
        $query->with(['installmentPurchaseRequest', 'installmentPurchaseRequest.user', 'installmentPurchaseRequest.plan', 'installmentPurchaseRequest.topic']);
        $query->where('is_overdue', 1);
        $query->latest();


        return Datatables::of($query)
            ->addIndexColumn()
            ->addColumn('name', function ($query) {
                return @$query->installmentPurchaseRequest->user->name;
            })
            ->editColumn('plan', function ($query) {
                return @$query->installmentPurchaseRequest->plan->getTranslation('title', auth()->user()->language_code);
            })
            ->editColumn('product', function ($query) {
                return @$query->installmentPurchaseRequest->topic->getTranslation('title', auth()->user()->language_code);
            })
            ->editColumn('amount', function ($query) {
                return getPriceFormat(@$query->amount);
            })
            ->editColumn('due_date', function ($query) {
                return showDate(@$query->due_date);
            })
            ->editColumn('status', function ($query) {
                return @$query->status;
            })
            ->editColumn('payment_date', function ($query) {
                return showDate(@$query->payment_date);
            })
            ->editColumn('delay', function ($query) {
                return Carbon::parse(@$query->due_date)->diffInDays(Carbon::parse(@$query->payment_date));
            })
            ->editColumn('action', function ($query) {
                return view('installment::overdue.action_menu', compact('query'));
            })->rawColumns(['name'])
            ->make(true);
    }

    public function verifiedUsersLoad(Request $request)
    {
        $query = User::query();
        $query->with(['installmentPurchases', 'installmentPurchases.plan', 'installmentPurchases.installmentPayments']);
        if (isModuleActive('LmsSaas')) {
            $query->where('lms_id', app('institute')->id);
        } else {
            $query->where('lms_id', 1);
        }
        $query->where('role_id', 3)->where('installment_verified', 1);

        $query->latest();

        return Datatables::of($query)
            ->addIndexColumn()
            ->addColumn('image', function ($query) {
                return view('backend.partials._td_image', compact('query'));
            })->editColumn('name', function ($query) {
                return $query->name;

            })->editColumn('total_purchase', function ($query) {
                return getPriceFormat($query->installmentPurchases->where('payment_status', '!=', 'incomplete')->sum('total_amount'));
            })
            ->editColumn('total_installment', function ($query) {
                $amount = $query->installmentPurchases->where('payment_status', '!=', 'incomplete')
                    ->sum(function ($q) {
                        return $q->installmentPayments
                            ->where('type', '!=', 'upfront')
                            ->sum('amount');
                    });
                return getPriceFormat($amount);
            })
            ->editColumn('overdue_installment', function ($query) {
                return $query->installmentPurchases->where('payment_status', '!=', 'incomplete')
                    ->sum(function ($q) {
                        return $q->installmentPayments
                            ->where('status', '!=', 'paid')
                            ->where('due_date', '<', date('Y-m-d'))
                            ->count();
                    });

            })
            ->editColumn('installment_count', function ($query) {
                return $query->installmentPurchases->where('payment_status', '!=', 'incomplete')
                    ->sum(function ($q) {
                        return $q->installmentPayments
                            ->where('type', '!=', 'upfront')
                            ->count();
                    });
            })->editColumn('action', function ($query) {
                return view('installment::verification.action_menu', compact('query'));

            })->rawColumns(['image'])
            ->make(true);
    }

    public function sentReminder($purchase_id)
    {
        try {
            $installment_payment = InstallmentPurchasePayment::where('id', $purchase_id)->with(['installmentPurchaseRequest', 'installmentPurchaseRequest.user', 'installmentPurchaseRequest.plan', 'installmentPurchaseRequest.topic'])->first();
            $purchase = $installment_payment->installmentPurchaseRequest;
            $user = $purchase->user;


            SendNotification::dispatch('Installment_Reminder', $user, [
                'name' => $user->name,
                'course' => $purchase->topic->getTranslation('title', auth()->user()->language_code),
                'plan' => $purchase->plan->getTranslation('title', auth()->user()->language_code),
                'due_date' => Carbon::parse($installment_payment->due_date)->format('jS M, Y'),
                'course_link' => courseDetailsUrl(@$purchase->topic->id, @$purchase->topic->type, @$purchase->topic->slug),
            ],[]);
            Toastr::success(_trans('installment.Installment Reminder Sent Successfully'), trans('common.Success'));
            return redirect()->route('installment.overdue_history');
        } catch (Throwable $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }
}
